/*
 * Copyright (c) 1990-1992 AT&T All Rights Reserved
 *	This is unpublished proprietary source code of AT&T
 *	The copyright notice above does not evidence any
 *	actual or intended publication of such source code.
 *
 * Pi Unix driver for Sun Sparc Workstations running Solaris 2.1
 *	This file uses /proc to examine running processes.
 *
 * D. A. Kapilow	3/3/93
 */
#include <stdio.h>
#include <signal.h>
#include <pwd.h>
#include <malloc.h>
#include <elf.h>
#include <time.h>
#include <sys/types.h>
#include <sys/syscall.h>
#include <sys/signal.h>
#include <sys/fault.h>
#include <sys/procfs.h>
#include <sys/wait.h>
#include <ulimit.h>
#include "dbmonitor/dbhost.h"

#define	bit(x)		(1<<(x-1))
#define	SPARCBPT	0x91D02001		/* t 1 */
#define	SPARC_TRAPMASK	0xDFF82000
#define	SPARC_TRAP	0x91D02000
#define REGSIZE		(36 * 4)
#define	REG_PC		1
#define	ISCORE(p)	((p)->pid < -1)

/*
 * Data structure private to each process or core dump
 */
struct coreseg {
	int fd;
	unsigned long	start;
	unsigned long	end;
	unsigned long	offset;
};

struct localproc {
	int		pid;
	int		corefd;
	int		opencnt;
	struct	Unixstate	state;
	struct localproc	*next;
	prstatus_t	s;
	/* For core dumps */
	int		stabfd;		/* text */
	int		nseg;
	struct coreseg	*seg;
};

/*
 * Register layout transmitted to pi
 */
struct dbregs {
	int r[19];
};

struct localproc *phead;
static char procname[] = "/proc/XXXXX";
static char procbuffer[128];
static sysset_t entrymask;
static int hangpid;
long pi_corepid = -1;
static unsigned long regaddr = -REGSIZE - 4;
static long bkpt_instr = SPARCBPT;
static char *pscmds[] = {
	"/bin/ps -u XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
	"/bin/ps -a    ",
	"/bin/ps -e    ",
	0
};

/* Routines local to this file */
static void state();
static void waithandler();
static char *regrw();
static void coreclose();
static char *corerw();

/* Also use for initialization */
char *pi_osname()
{
	struct passwd *pw;
	static int first = 1;

	if (first) {
		close(2); /* Otherwise error messages from ps goto rtpi */
		open("/dev/null", 2);
		pw = getpwuid(getuid());
		if (pw)
			sprintf(pscmds[0], "/bin/ps -u %s", pw->pw_name);
		else
			sprintf(pscmds[0], "/bin/ps");
		signal(SIGCLD, waithandler);
		first = 0;
	}
	return "SPARC SunOS 5.1";
}

char	**pi_getpscmds()	{ return pscmds; }
int	pi_getpsfield()		{ return 4; }
long	pi_regaddr()		{ return (long)regaddr; }
long	pi_scratchaddr()	{ return 0x10000; }
int	pi_nsig()		{ return NSIG; }
int	pi_exechangsupported()	{ return 1; }
int	pi_stabfdsupported()	{ return 1; }

long pi_sigmaskinit()
{
	return	bit(SIGILL)|bit(SIGINT)|bit(SIGTRAP)|bit(SIGIOT)|
		bit(SIGEMT)|bit(SIGFPE)|bit(SIGBUS)|bit(SIGSEGV)|
		bit(SIGSYS)|bit(SIGPIPE)|bit(SIGSTOP);
}

struct localproc *pi_pidtoproc(pid)
{
	register struct localproc *p;

	for (p = phead; p; p = p->next)
		if (p->pid == pid)
			return p;
	return 0;
}

struct localproc *pi_open(pid)
{
	register struct localproc *p = phead;
	sysset_t sysmask;
	int wasactive = 0;
	long mode;

	if (pid < 0)
		return 0;
	if (p = pi_pidtoproc(pid)) {
		p->opencnt++;
		return p;
	}
	p = (struct localproc *)malloc(sizeof(struct localproc));
	if (!p)
		return 0;
	sprintf(procname,"/proc/%05d", pid);
	if ((p->corefd = open(procname, 2)) < 0) {
		free(p);
		return 0;
	}
	p->pid = pid;
	p->opencnt = 1;
	p->next = phead;
	phead = p;
	state(p);
	premptyset(&sysmask);
	ioctl(p->corefd, PIOCSENTRY, &sysmask);
	ioctl(p->corefd, PIOCSEXIT, &sysmask);
	mode = PR_FORK;
	ioctl(p->corefd, PIOCSET, &mode);
	return p;
}

void pi_close(p)
struct localproc *p;
{
	register struct localproc *q;
	long mode;
	sysset_t sysmask;
	sigset_t sset;

	if (--p->opencnt > 0)
		return;
	if (ISCORE(p))
		coreclose(p);
	else {
		if (p->state.state != UNIX_ERRORED) {
			premptyset(&sysmask);
			ioctl(p->corefd, PIOCSENTRY, &sysmask);
			ioctl(p->corefd, PIOCSEXIT, &sysmask);
			premptyset(&sset);
			ioctl(p->corefd, PIOCSTRACE, &sset);
			mode = PR_RLC | PR_FORK;
			ioctl(p->corefd, PIOCRESET, &mode);
		}
		close(p->corefd);
	}
	if (p == phead)
		phead = p->next;
	else {
		for (q = phead; q->next != p; q = q->next)
			;
		q->next = p->next;
	}
	free(p);
}

int pi_getsymtabfd(p)
struct localproc *p;
{
	return ioctl(p->corefd, PIOCOPENM, 0);
}

char *pi_run(p)
struct localproc *p;
{
	ioctl(p->corefd, PIOCRUN, 0);
	return 0;
}

char *pi_stop(p)
struct localproc *p;
{
	ioctl(p->corefd, PIOCSTOP, 0);
	return 0;
}

char *pi_destroy(p)
struct localproc *p;
{
	int sig = SIGKILL;
	ioctl(p->corefd, PIOCKILL, &sig);
	state(p);
	p->state.state = UNIX_ERRORED;
	p->state.code = SIGKILL;
	return 0;
}

void pi_getstate(p, s)
struct localproc *p;
struct	Unixstate *s;
{
	if (!ISCORE(p))
		state(p);
	*s = p->state;
}

static void state(p)
struct localproc *p;
{
	if (ioctl(p->corefd, PIOCSTATUS, &p->s)) {
		/*
		 * If the call failed, it is probably because the process has
		 * disappeared. With /proc, there is no way to
		 * determine the exist status or reason the program died, so
		 * set it to make it look like a signal that cannot really
		 * happen was sent. If the server is the parent of the process,
		 * handling the SIGCLD signal will set the correct state value.
		 * Otherwise, the setting below remains.
		 */
		if (p->state.state != UNIX_ERRORED) {
			p->state.state = UNIX_ERRORED;
			p->state.code = NSIG;
		}
		return;
	}
	if (!(p->s.pr_flags & PR_STOPPED))
		p->state.state = UNIX_ACTIVE;
	else if (p->s.pr_why == PR_SIGNALLED) {
		if (p->s.pr_what == SIGTRAP) {
			p->state.state = UNIX_BREAKED;
			ioctl(p->corefd, PIOCSSIG, 0);
		} else {
			p->state.state = UNIX_PENDING;
			p->state.code = p->s.pr_what;
		}
	} else
		p->state.state = UNIX_HALTED;
}

char *pi_readwrite(p, buf, addr, r, w)
struct localproc *p;
char *buf;
unsigned long addr;
{
	int count, ret;

	if (ISCORE(p))
		return corerw(p, buf, addr, r, w);
	if (addr >= regaddr)
		return regrw(p, buf, (int)(addr - regaddr), r, w);
	if (lseek(p->corefd, addr, SEEK_SET) == -1)
		return "lseek on /proc failed";
	if (w) {
		ret = write(p->corefd, buf, w);
		count = w;
	} else {
		ret = read(p->corefd, buf, r);
		count = r;
	}
	if (ret != count)
		 return "read/write on /proc failed";
	return 0;
}

static char *regrw(p, buf, offset, r, w)
struct localproc *p;
char *buf;
{
	struct dbregs reg;
	prgregset_t procreg;
	char *error = 0;
	int wasactive = 0;

	if (p->state.state == UNIX_ACTIVE) {
		ioctl(p->corefd, PIOCSTOP, 0);
		state(p);
		if (p->s.pr_why == PR_REQUESTED)
			wasactive = 1;
	}
	if (ioctl(p->corefd, PIOCGREG, &procreg)) {
		error =  "DFCGETREGS failed";
		goto out;
	}
	memcpy((char*)&reg.r[0], (char*)&procreg[R_PSR], 4*4);
	memcpy((char*)&reg.r[4], (char*)&procreg[R_G1], 7*4);
	memcpy((char*)&reg.r[11], (char*)&procreg[R_O0], 8*4);
	if (w) {
		memcpy((char*)&reg + offset, buf, w);
		memcpy((char*)&procreg[R_PSR], (char*)&reg.r[0], 4*4);
		memcpy((char*)&procreg[R_G1], (char*)&reg.r[4], 7*4);
		memcpy((char*)&procreg[R_O0], (char*)&reg.r[11], 8*4);
		ioctl(p->corefd, PIOCSREG, &procreg);
	} else
		memcpy(buf, (char*)&reg + offset, r);
out:
	if (wasactive)
		ioctl(p->corefd, PIOCRUN, 0);
	return error;
}

char *pi_setbkpt(p, addr)
struct localproc *p;
long addr;
{
	return pi_readwrite(p,(char*)&bkpt_instr, addr, 0, sizeof(bkpt_instr));
}

char *pi_step(p)
struct localproc *p;
{
	prrun_t prrun;

	prrun.pr_flags = PRSTEP;
	if (ioctl(p->corefd, PIOCRUN, &prrun))
		return "DFCSSTEP error";
	if (ioctl(p->corefd, PIOCWSTOP, 0))
		return "DFCWSTOP error";
	return 0;
}

static void alarmcatcher(int i)	{}

char *pi_waitstop(p)
struct localproc *p;
{
	char *err = 0;
	void (*save)();

	save = signal(SIGALRM, alarmcatcher);
	alarm(15);
	if (ioctl(p->corefd, PIOCWSTOP, 0))
		err = "timeout waiting for breakpoint";
	alarm(0);
	signal(SIGALRM, save);
	return err;
}

char *pi_ssig(p, sig)
struct localproc *p;
long sig;
{
	ioctl(p->corefd, PIOCKILL, &sig);
	return 0;
}

char *pi_csig(p)
struct localproc *p;
{
	ioctl(p->corefd, PIOCSSIG, 0);
	return 0;
}

char *pi_sigmask(p, mask)
struct localproc *p;
long mask;
{
	sigset_t sset;
	int i;

	premptyset(&sset);
	for(i = 1; i <= 32; i++)
		if (mask & bit(i))
			praddset(&sset, i);
	ioctl(p->corefd, PIOCSTRACE, &sset);
	return 0;
}

char *pi_signalname(sig)
long sig;
{
	char *cp;

	switch (sig) {
	case SIGHUP:	cp = "hangup"; break;
	case SIGINT:	cp = "interrupt"; break;
	case SIGQUIT:	cp = "quit"; break;
	case SIGILL:	cp = "illegal instruction"; break;
	case SIGTRAP:	cp = "trace trap"; break;
	case SIGABRT:	cp = "abort"; break;
	case SIGEMT:	cp = "EMT instruction"; break;
	case SIGFPE:	cp = "floating exception"; break;
	case SIGKILL:	cp = "kill"; break;
	case SIGBUS:	cp = "bus error"; break;
	case SIGSEGV:	cp = "segmentation violation"; break;
	case SIGSYS:	cp = "bad system call"; break;
	case SIGPIPE:	cp = "broken pipe"; break;
	case SIGALRM:	cp = "alarm call"; break;
	case SIGTERM:	cp = "terminated"; break;
	case SIGUSR1:	cp = "user signal 1"; break;
	case SIGUSR2:	cp = "user signal 2"; break;
	case SIGCHLD:	cp = "child termination"; break;
	case SIGPWR:	cp = "power-fail"; break;
	case SIGWINCH:	cp = "window changed"; break;
	case SIGURG:	cp = "urgent i/o condition"; break;
	case SIGPOLL:	cp = "pollable event"; break;
	case SIGSTOP:	cp = "stop"; break;
	case SIGTSTP:	cp = "tty stop"; break;
	case SIGCONT:	cp = "continue"; break;
	case SIGTTIN:	cp = "stop tty input"; break;
	case SIGTTOU:	cp = "stop tty output"; break;
	case SIGVTALRM:	cp = "virtual time alarm"; break;
	case SIGPROF:	cp = "profiling alarm"; break;
	case SIGXCPU:	cp = "cpu time limit"; break;
	case SIGXFSZ:	cp = "file size limit"; break;
	case NSIG:	cp = "cannot determine cause"; break;
	default:
		cp = procbuffer;
		sprintf(procbuffer, "signal %d", sig);
		break;
	}
	return cp;
}

char *pi_proctime(p)
struct localproc *p;
{
	sprintf(procbuffer, "%d.%02du %d.%02ds",
		p->s.pr_utime.tv_sec, p->s.pr_utime.tv_nsec/10000000,
		p->s.pr_stime.tv_sec, p->s.pr_stime.tv_nsec/10000000);
	return procbuffer;
}

int pi_getppid(p)
struct localproc *p;
{
	return (int)p->s.pr_ppid;
}

int pi_atsyscall(p)
struct localproc *p;
{
	long pc, instr; 

	pi_readwrite(p, (char *)&pc, regaddr + REG_PC * 4, sizeof(pc), 0);
	pi_readwrite(p, (char *)&instr, pc, sizeof(instr), 0);
	return ((instr & SPARC_TRAPMASK) == SPARC_TRAP);
}

char *pi_exechang(p, e)
struct localproc *p;
long e;
{
	sysset_t sysmask;

	ioctl(p->corefd, PIOCGEXIT, &sysmask);
	if (e) {
		praddset(&sysmask, SYS_exec);
		praddset(&sysmask, SYS_execve);
	}
	else {
		prdelset(&sysmask, SYS_exec);
		prdelset(&sysmask, SYS_execve);
	}
	ioctl(p->corefd, PIOCSEXIT, &sysmask);
	return 0;
}

/*
 * Expand a path
 */
static char *pathexpand(f, path, a)
char *f, *path;
int a;
{
	static char file[128];
	register char *p;

	if (*f != '/' && strncmp(f, "./", 2) && strncmp(f, "../", 3) && 
	    path!=0){
		while(*path){
			for(p=file; *path && *path!=':';)
				*p++ = *path++;
			if(p!=file)
				*p++='/';
			if(*path)
				path++;
			(void)strcpy(p, f);
			if (access(file, a) != -1)
				return file;
		}
	}
	if (access(f, a) != -1 )
		return f;
	return 0;
}

/*
 * This function and the function below are for starting new
 * processes from pi.
 */
int pi_hang(cmd)
char *cmd;
{
	char *argv[10], *cp;
	int i, fd;
	char *file, *getenv();
	struct localproc *p;
	sysset_t sysmask;
	long mode;
	int nfiles;
	
	i = strlen(cmd);
	if (++i > sizeof(procbuffer)) {
		i = sizeof(procbuffer) - 1;
		procbuffer[i] = 0;
	}
	memcpy(procbuffer, cmd, i);
	argv[0] = cp = procbuffer;
	for(i = 1;;) {
		while(*cp && *cp != ' ')
			cp++;
		if (!*cp) {
			argv[i] = 0;
			break;
		} else {
			*cp++ = 0;
			while (*cp == ' ')
				cp++;
			if (*cp)
				argv[i++] = cp;
		}
	}
	if (!(file = pathexpand(argv[0], getenv("PATH"), 5)))
		return 0;
	hangpid = fork();
	if (!hangpid){
		nfiles = ulimit(UL_GDESLIM,0);
		for (fd = 0; fd < nfiles; ++fd)
			close(fd);
		open("/dev/null", 2);
		dup2(0, 1);
		dup2(0, 2);
		setpgrp(0, getpid());
		sprintf(procname, "/proc/%05d", getpid());
		if ((fd = open(procname, 2)) < 0)
			exit(1);
		mode = PR_RLC;
		ioctl(fd, PIOCRESET, &mode);
		premptyset(&sysmask);
		praddset(&sysmask, SYS_exec);
		praddset(&sysmask, SYS_execve);
		ioctl(fd, PIOCSEXIT, &sysmask);
		close(fd);
		execv(file, argv);
		exit(0);
	}
	if (hangpid < 0)
		return 0;
	sprintf(procname,"/proc/%05d", hangpid);
	if ((fd = open(procname, 2)) < 0) {
		kill(9, hangpid);
		return 0;
	}
	mode = PR_RLC;
	ioctl(fd, PIOCRESET, &mode);
	p = (struct localproc *)malloc(sizeof(struct localproc));
	if (!p) {
		kill(9, hangpid);
		close(fd);
		return 0;
	}
	p->corefd = fd;
	p->pid = hangpid;
	p->opencnt = 0;
	p->next = phead;
	phead = p;
	ioctl(p->corefd, PIOCWSTOP, 0);
	state(p);
	/* If the exec failed - get rid of it */
/*
	if (p->s.pr_errno) {
		pi_ssig(p, 9);
		state(p);
		pi_close(p);
		return 0;
	}
*/
	premptyset(&sysmask);
	ioctl(p->corefd, PIOCSENTRY, &sysmask);
	ioctl(p->corefd, PIOCSEXIT, &sysmask);
	mode = PR_FORK;
	ioctl(p->corefd, PIOCSET, &mode);
	return hangpid;
}

static void waithandler(int i)
{
	int pid, cursig;
	int tstat;
	struct localproc *p;

	pid = wait(&tstat);
	signal(SIGCLD, waithandler);
	if (pid < 0 || !(p = pi_pidtoproc(pid)))
		return;
	if (WIFSTOPPED(tstat)) {
		cursig = WSTOPSIG(tstat);
		if (cursig == SIGSTOP)
			p->state.state = UNIX_HALTED;
		else if (cursig == SIGTRAP)
			p->state.state = UNIX_BREAKED;
		else {
			p->state.state = UNIX_PENDING;
			p->state.code = cursig;
		}
	} else {
		p->state.state = UNIX_ERRORED;
		p->state.code = tstat & 0xFFFF;
	}
}

/*
 * The functions below are for debugging core dumps
 */
struct localproc *pi_coreopen(corep, symtabp)
char *corep, *symtabp;
{
	register struct localproc *p;
	register struct coreseg *cs;
	int symtabfd, corefd = -1;
	int mode;
	int i, n, n2, note = -1;
	Elf32_Ehdr chdr, shdr;
	Elf32_Phdr *cphdr, *sphdr;
	char *notep, *np;
	long *lp;
	int statusfound = 0;

	if ((symtabfd = open(symtabp, 0)) < 0)
		return 0;
	for (mode = 2; corefd < 0 && mode >= 0; mode--)
		corefd = open(corep, mode);
	if (corefd < 0 ||
	    read(corefd, (char*)&chdr, sizeof(chdr)) != sizeof(chdr) ||
	    memcmp(&chdr.e_ident[EI_MAG1], "ELF", 3) ||
	    chdr.e_ident[EI_CLASS] != ELFCLASS32 ||
	    chdr.e_ident[EI_DATA] != ELFDATA2MSB ||
	    chdr.e_type != ET_CORE ||
	    chdr.e_machine != EM_SPARC ||
	    !chdr.e_phnum ||
	    symtabfd < 0 ||
	    read(symtabfd, (char*)&shdr, sizeof(shdr)) != sizeof(shdr) ||
	    memcmp(&shdr.e_ident[EI_MAG1], "ELF", 3) ||
	    shdr.e_ident[EI_CLASS] != ELFCLASS32 ||
	    shdr.e_ident[EI_DATA] != ELFDATA2MSB ||
	    shdr.e_type != ET_EXEC ||
	    shdr.e_machine != EM_SPARC ||
	    !shdr.e_phnum) {
		close(corefd);
		close(symtabfd);
		return 0;
	}

	p = (struct localproc *)malloc(sizeof(struct localproc));
	n = sizeof(Elf32_Phdr) * chdr.e_phnum;
	n2 = sizeof(Elf32_Phdr) * shdr.e_phnum;
	cphdr = (Elf32_Phdr *)malloc(n);
	sphdr = (Elf32_Phdr *)malloc(n2);
	if (!p || !cphdr || !sphdr ||
	    lseek(corefd, (long)chdr.e_phoff, SEEK_SET) == -1 ||
	    read(corefd, (char*)cphdr, n) != n ||
	    lseek(symtabfd, (long)shdr.e_phoff, SEEK_SET) == -1 ||
	    read(symtabfd, (char*)sphdr, n2) != n2) {
		if (p) free(p);
		if (cphdr) free((char*)cphdr);
		if (sphdr) free((char*)sphdr);
		close(corefd);
		close(symtabfd);
		return 0;
	}
	p->pid = --pi_corepid;
	p->opencnt = 1;
	p->next = phead;
	phead = p;
	p->corefd = corefd;
	p->stabfd = symtabfd;
	p->nseg = 0;

	for(i = 0; i < (int)chdr.e_phnum; i++) {
		if (cphdr[i].p_type == PT_NOTE)
			note = i;
		if (cphdr[i].p_type == PT_LOAD && cphdr[i].p_filesz)
			p->nseg++;
	}
	/* One for text segment from symtab file */
	p->nseg++;
	n = cphdr[note].p_filesz;
	if (note == -1 || !(notep = malloc(n))) {
		free(p);
		free(cphdr);
		free(sphdr);
		close(corefd);
		close(symtabfd);
		return 0;
	}
	lseek(corefd, cphdr[note].p_offset, SEEK_SET);
	read(corefd, notep, n);
	for(np = notep; np < &notep[n]; ) {
		lp = (long *)np;
		np += 12;
		np += ((lp[0] + 3) >> 2) << 2;
		if (lp[2] == 1 && lp[1] == sizeof(prstatus_t)) {
			p->s = *(prstatus_t *)np;
			statusfound = 1;
		}
		np += ((lp[1] + 3) >> 2) << 2;
	}
	free(notep);
	p->seg = cs = (struct coreseg *)
			malloc(sizeof(struct coreseg) * p->nseg);
	if (!cs || !statusfound) {
		if (cs)
			free(cs);
		free(p);
		free(cphdr);
		free(sphdr);
		close(corefd);
		close(symtabfd);
		return 0;
	}
	for(i = 0; i < (int)chdr.e_phnum; i++) {
		if (cphdr[i].p_type == PT_LOAD && cphdr[i].p_filesz) {
			cs->start = cphdr[i].p_vaddr;
			cs->end = cs->start + cphdr[i].p_filesz;
			cs->offset = cphdr[i].p_offset;
			cs->fd = corefd;
			cs++;
		}
	}
	for(i = 0; i < (int)shdr.e_phnum; i++) {
		if (sphdr[i].p_type == PT_LOAD && sphdr[i].p_filesz &&
		    sphdr[i].p_flags == (PF_R|PF_X)) {
			cs->start = sphdr[i].p_vaddr;
			cs->end = cs->start + sphdr[i].p_filesz;
			cs->offset = sphdr[i].p_offset;
			cs->fd = symtabfd;
			cs++;
			break;
		}
	}
	free(cphdr);
	free(sphdr);
	p->state.state = UNIX_PENDING;
	p->state.code = p->s.pr_cursig;
	return p;
}

static void coreclose(p)
struct localproc *p;
{
	close(p->corefd);
	close(p->stabfd);
	free(p->seg);
}

static char *coreregrw(p, buf, offset, r, w)
struct localproc *p;
char *buf;
{
	struct dbregs reg;
	prgregset_t procreg;

	memcpy((char*)&reg.r[0], (char*)&p->s.pr_reg[R_PSR], 4*4);
	memcpy((char*)&reg.r[4], (char*)&p->s.pr_reg[R_G1], 7*4);
	memcpy((char*)&reg.r[11], (char*)&p->s.pr_reg[R_O0], 8*4);
	if (w) {
		memcpy((char*)&reg + offset, buf, w);
		memcpy((char*)&p->s.pr_reg[R_PSR], (char*)&reg.r[0], 4*4);
		memcpy((char*)&p->s.pr_reg[R_G1], (char*)&reg.r[4], 7*4);
		memcpy((char*)&p->s.pr_reg[R_O0], (char*)&reg.r[11], 8*4);
	} else
		memcpy(buf, (char*)&reg + offset, r);
	return 0;
}

static char *corerw(p, buf, addr, r, w)
struct localproc *p;
char *buf;
unsigned long addr;
{
	register struct coreseg *cs, *csend;

	if (addr >= regaddr)
		return coreregrw(p, buf, (int)(addr - regaddr), r, w);
	cs = p->seg;
	csend = &cs[p->nseg];
	for (; cs < csend; cs++)
		if (cs->start <= addr && addr < cs->end)
			break;
	if (cs == csend)
		return "corerw:invalid address";
	addr = addr - cs->start + cs->offset;
	if (lseek(cs->fd, addr, SEEK_SET) == -1)
		return "corerw:lseek failed";
	if (w) {
		if (write(cs->fd, buf, w) != w)
			return "corerw:write error";
	} else {
		if (read(cs->fd, buf, r) != r)
			return "corerw:read error";
	}
	return 0;
}
